<?php

class KnowledgeManager
{
    private $config;
    private $storageDir;
    private $elevenlabsService;
    
    public function __construct($config, $elevenlabsService)
    {
        $this->config = $config;
        $this->elevenlabsService = $elevenlabsService;
        $this->storageDir = __DIR__ . '/../../storage/knowledge_base/';
        
        if (!is_dir($this->storageDir)) {
            mkdir($this->storageDir, 0755, true);
        }
    }
    
    /**
     * Add knowledge base content
     */
    public function addContent($content, $category, $title, $type = 'text', $source = null)
    {
        try {
            // Validate category
            $allowedCategories = array_keys($this->config['knowledge_base']['categories']);
            if (!in_array($category, $allowedCategories)) {
                throw new Exception('Geçersiz kategori');
            }
            
            // Save to local storage first
            $contentId = $this->saveToStorage($content, $category, $title, $type, $source);
            
            // Try to add to Elevenlabs knowledge base (optional)
            $elevenlabsResponse = null;
            try {
                $elevenlabsResponse = $this->elevenlabsService->addKnowledgeBase(
                    $content, 
                    $type, 
                    $category
                );
                
                // Update local record with Elevenlabs ID if successful
                $this->updateLocalRecord($contentId, $elevenlabsResponse);
                
            } catch (Exception $elevenlabsError) {
                // Log the error but don't fail the entire operation
                error_log('Elevenlabs Knowledge Base Error (non-critical): ' . $elevenlabsError->getMessage());
            }
            
            return [
                'success' => true,
                'content_id' => $contentId,
                'elevenlabs_id' => $elevenlabsResponse['id'] ?? null,
                'message' => 'İçerik başarıyla eklendi' . (($elevenlabsResponse['id'] ?? null) ? '' : ' (Elevenlabs senkronizasyonu daha sonra yapılacak)')
            ];
            
        } catch (Exception $e) {
            error_log('Knowledge Manager Error: ' . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Update knowledge base content
     */
    public function updateContent($contentId, $newContent)
    {
        try {
            // Get local record
            $record = $this->getLocalRecord($contentId);
            if (!$record) {
                throw new Exception('İçerik bulunamadı');
            }
            
            // Update in Elevenlabs
            if ($record['elevenlabs_id']) {
                $this->elevenlabsService->updateKnowledgeBase(
                    $record['elevenlabs_id'], 
                    $newContent
                );
            }
            
            // Update local storage
            $record['content'] = $newContent;
            $record['updated_at'] = date('Y-m-d H:i:s');
            $this->saveLocalRecord($contentId, $record);
            
            return [
                'success' => true,
                'message' => 'İçerik başarıyla güncellendi'
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Delete knowledge base content
     */
    public function deleteContent($contentId)
    {
        try {
            $record = $this->getLocalRecord($contentId);
            if (!$record) {
                throw new Exception('İçerik bulunamadı');
            }
            
            // Delete from Elevenlabs
            if ($record['elevenlabs_id']) {
                $this->elevenlabsService->deleteKnowledgeBase($record['elevenlabs_id']);
            }
            
            // Delete local files
            $this->deleteLocalRecord($contentId);
            
            return [
                'success' => true,
                'message' => 'İçerik başarıyla silindi'
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Get all knowledge base content
     */
    public function getAllContent($category = null)
    {
        $content = [];
        $categories = $category ? [$category] : array_keys($this->config['knowledge_base']['categories']);
        
        foreach ($categories as $cat) {
            $categoryDir = $this->storageDir . $cat . '/';
            if (is_dir($categoryDir)) {
                $files = glob($categoryDir . '*.json');
                foreach ($files as $file) {
                    $record = json_decode(file_get_contents($file), true);
                    if ($record) {
                        $content[] = $record;
                    }
                }
            }
        }
        
        return $content;
    }
    
    /**
     * Search in knowledge base
     */
    public function searchContent($query, $category = null)
    {
        $allContent = $this->getAllContent($category);
        $results = [];
        
        $query = strtolower($query);
        
        foreach ($allContent as $content) {
            $searchText = strtolower($content['title'] . ' ' . $content['content']);
            if (strpos($searchText, $query) !== false) {
                $results[] = $content;
            }
        }
        
        return $results;
    }
    
    /**
     * Upload file content
     */
    public function uploadFile($file, $category, $title = null)
    {
        try {
            // Validate file
            $this->validateFile($file);
            
            // Extract content from file
            $content = $this->extractFileContent($file);
            
            $title = $title ?: pathinfo($file['name'], PATHINFO_FILENAME);
            
            return $this->addContent($content, $category, $title, 'file', $file['name']);
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Add URL content
     */
    public function addUrlContent($url, $category, $title = null)
    {
        try {
            // Fetch URL content
            $content = $this->fetchUrlContent($url);
            
            $title = $title ?: parse_url($url, PHP_URL_HOST);
            
            return $this->addContent($content, $category, $title, 'url', $url);
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Get predefined real estate knowledge templates
     */
    public function getTemplates()
    {
        return [
            'genel_bilgiler' => [
                'title' => 'Genel Gayrimenkul Bilgileri',
                'content' => $this->getGeneralRealEstateInfo()
            ],
            'satis_surecler' => [
                'title' => 'Satış Süreçleri',
                'content' => $this->getSalesProcessInfo()
            ],
            'kira_surecler' => [
                'title' => 'Kiralama Süreçleri',
                'content' => $this->getRentalProcessInfo()
            ],
            'kredi_mortgage' => [
                'title' => 'Kredi ve Mortgage',
                'content' => $this->getMortgageInfo()
            ]
        ];
    }
    
    /**
     * Initialize default knowledge base
     */
    public function initializeDefaults()
    {
        $templates = $this->getTemplates();
        $results = [];
        
        foreach ($templates as $category => $template) {
            $result = $this->addContent(
                $template['content'],
                $category,
                $template['title'],
                'text'
            );
            $results[$category] = $result;
        }
        
        return $results;
    }
    
    // Private helper methods
    
    private function saveToStorage($content, $category, $title, $type, $source)
    {
        $categoryDir = $this->storageDir . $category . '/';
        if (!is_dir($categoryDir)) {
            mkdir($categoryDir, 0755, true);
        }
        
        $contentId = uniqid();
        $record = [
            'id' => $contentId,
            'content_id' => $contentId, // Add this for consistency with frontend
            'title' => $title,
            'content' => $content,
            'category' => $category,
            'type' => $type,
            'source' => $source,
            'created_at' => date('Y-m-d H:i:s'),
            'updated_at' => date('Y-m-d H:i:s'),
            'elevenlabs_id' => null
        ];
        
        $filename = $categoryDir . $contentId . '.json';
        file_put_contents($filename, json_encode($record, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
        
        return $contentId;
    }
    
    private function updateLocalRecord($contentId, $elevenlabsResponse)
    {
        $record = $this->getLocalRecord($contentId);
        if ($record) {
            $record['elevenlabs_id'] = $elevenlabsResponse['id'] ?? null;
            $record['updated_at'] = date('Y-m-d H:i:s');
            $this->saveLocalRecord($contentId, $record);
        }
    }
    
    private function getLocalRecord($contentId)
    {
        foreach (array_keys($this->config['knowledge_base']['categories']) as $category) {
            $filename = $this->storageDir . $category . '/' . $contentId . '.json';
            if (file_exists($filename)) {
                return json_decode(file_get_contents($filename), true);
            }
        }
        return null;
    }
    
    private function saveLocalRecord($contentId, $record)
    {
        $categoryDir = $this->storageDir . $record['category'] . '/';
        $filename = $categoryDir . $contentId . '.json';
        file_put_contents($filename, json_encode($record, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    }
    
    private function deleteLocalRecord($contentId)
    {
        foreach (array_keys($this->config['knowledge_base']['categories']) as $category) {
            $filename = $this->storageDir . $category . '/' . $contentId . '.json';
            if (file_exists($filename)) {
                unlink($filename);
                return true;
            }
        }
        return false;
    }
    
    private function validateFile($file)
    {
        $allowedFormats = $this->config['knowledge_base']['allowed_formats'];
        $maxSize = $this->config['knowledge_base']['max_file_size'];
        
        if ($file['size'] > $maxSize) {
            throw new Exception('Dosya boyutu çok büyük (Max: ' . round($maxSize/1024/1024) . 'MB)');
        }
        
        $extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        if (!in_array($extension, $allowedFormats)) {
            throw new Exception('Desteklenmeyen dosya formatı. İzin verilenler: ' . implode(', ', $allowedFormats));
        }
    }
    
    private function extractFileContent($file)
    {
        $extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        
        switch ($extension) {
            case 'txt':
                return file_get_contents($file['tmp_name']);
            case 'html':
                return strip_tags(file_get_contents($file['tmp_name']));
            case 'pdf':
                // PDF extraction would require additional libraries
                throw new Exception('PDF extraction henüz desteklenmiyor');
            case 'docx':
                // DOCX extraction would require additional libraries
                throw new Exception('DOCX extraction henüz desteklenmiyor');
            default:
                throw new Exception('Desteklenmeyen dosya formatı');
        }
    }
    
    private function fetchUrlContent($url)
    {
        $content = file_get_contents($url);
        if ($content === false) {
            throw new Exception('URL içeriği alınamadı');
        }
        
        return strip_tags($content);
    }
    
    // Knowledge base templates
    
    private function getGeneralRealEstateInfo()
    {
        return "
GENEL GAYRİMENKUL BİLGİLERİ

Gayrimenkul Türleri:
- Konut: Daire, villa, müstakil ev, dubleks
- Ticari: Dükkan, ofis, depo, fabrika
- Arsa: İmarlı arsa, tarla, bahçe

Tapu Türleri:
- Kat Mülkiyeti: En yaygın tapu türü
- Kat İrtifakı: Yapı kullanma izni alınmadan önce
- Arsa Payı: Arsanın bölünmemiş payı

Değerleme Kriterleri:
- Konum ve ulaşım
- Yapı yaşı ve durumu
- Metrekare bilgileri
- Kat sayısı ve bulunduğu kat
- Çevre düzenlemesi
- Sosyal tesisler

Önemli Belgeler:
- Tapu senedi
- Yapı kullanma izni
- Kat irtifakı/kat mülkiyeti krokisi
- Emlak vergisi beyannamesi
- Aidat durumu (site için)
        ";
    }
    
    private function getSalesProcessInfo()
    {
        return "
SATIŞ SÜREÇLERİ

Satış Öncesi Hazırlık:
1. Emlak değerleme raporu alma
2. Gerekli belgelerin teminı
3. Emlak fotoğrafları çekimi
4. Pazarlama stratejisi belirleme

Satış Süreci:
1. İlk görüşme ve ihtiyaç analizi
2. Uygun gayrimenkul önerisi
3. Gezdirim ve detaylı bilgilendirme
4. Fiyat pazarlığı
5. Ön protokol imzalama
6. Tapu devir işlemi

Gerekli Belgeler:
- Satıcı: Tapu, kimlik, vergi levhası
- Alıcı: Kimlik, gelir belgesi, kredi onayı (varsa)

Maliyetler:
- Tapu harcı (%4 - alıcı öder)
- Emlak komisyonu (%2-3)
- Ekspertiz ücreti
- Noter masrafları

Önemli Noktalar:
- Tapu kontrolü mutlaka yapılmalı
- İpotek, haciz durumu araştırılmalı
- Yapı kullanma izni kontrol edilmeli
- Aidat borcu sorgulanmalı
        ";
    }
    
    private function getRentalProcessInfo()
    {
        return "
KİRALAMA SÜREÇLERİ

Kiralama Öncesi:
1. Kiraya verilecek gayrimenkulün değerlendirilmesi
2. Uygun kira bedelinin belirlenmesi
3. Kira sözleşmesi hazırlığı
4. Kiracı profili belirleme

Kiralama Süreci:
1. İlk görüşme ve ihtiyaç belirleme
2. Uygun gayrimenkul önerisi
3. Gezdirim ve bilgilendirme
4. Referans kontrolü
5. Kira sözleşmesi imzalama
6. Depozito ve ön ödeme alımı

Kira Sözleşmesi:
- Minimum 1 yıl süreli olmalı
- Kira artış oranları belirtilmeli
- Depozito miktarı (genelde 1-2 aylık kira)
- Tarafların hak ve sorumlulukları

Kiracı Hakları:
- Kiranın zamanında ödenmesi
- Gayrimenkulü kontrat şartlarına uygun kullanma
- Hasar durumunda kiralayana bildirme

Kiralayan Hakları:
- Düzenli kira geliri
- Gayrimenkulün korunması
- Sözleşme bitiminde boşaltma

Yasal Düzenlemeler:
- TBK (Türk Borçlar Kanunu)
- Kira artış oranları (TÜFE)
- Tahliye süreçleri
        ";
    }
    
    private function getMortgageInfo()
    {
        return "
KREDİ VE MORTGAGE BİLGİLERİ

Mortgage Kredisi:
- Gayrimenkul alımında kullanılan uzun vadeli kredi
- Kredi tutarı gayrimenkul değerinin %80'ine kadar
- Vade 10-15 yıl arası
- Faiz oranları değişken veya sabit

Gerekli Belgeler:
- Kimlik fotokopisi
- Gelir belgesi (maaş bordrosu, SGK prim dökümü)
- İkametgah belgesi
- Alınacak gayrimenkulün değerleme raporu
- Banka hesap ekstreleri

Kredi Başvuru Süreci:
1. Ön değerlendirme (online veya şubede)
2. Belge teslimi
3. Ekspertiz raporu
4. Kredi onay süreci
5. Sözleşme imzalama
6. Tapu devir işlemi

Kredi Maliyetleri:
- Dosya masrafı
- Ekspertiz ücreti
- Sigorta primleri (dask, yangın, hayat)
- Komisyon oranları

Faiz Türleri:
- Değişken faiz: TCMB politika faizine bağlı
- Sabit faiz: Belirli dönem için sabit oran
- Karma faiz: İlk yıllar sabit, sonrası değişken

Erken Ödeme:
- Kredinin vadesinden önce kapatılması
- Erken ödeme cezası uygulanabilir
- Refinansman imkanları

Önemli Noktalar:
- Aylık gelirin 1/3'ünden fazla taksit olmamalı
- Kredi notu yüksek olmalı
- İstikrarlı gelir şartı
- Peşinat oranı yüksek olursa şartlar iyileşir
        ";
    }
} 