<?php

class ElevenlabsService
{
    private $config;
    private $apiKey;
    private $baseUrl;
    private $apiBase;
    private $agentId;
    private $voiceId;
    private $wsUrl;
    
    public function __construct($config)
    {
        $this->config = $config;
        $this->apiKey = $config['elevenlabs']['api_key'] ?? '';
        $this->baseUrl = $config['elevenlabs']['base_url'] ?? 'https://api.elevenlabs.io/v1';
        $this->apiBase = $this->baseUrl;
        $this->wsUrl = $config['elevenlabs']['ws_url'] ?? 'wss://api.elevenlabs.io/v1/convai/conversation';
        $this->agentId = $config['elevenlabs']['agent_id'] ?? '';
        $this->voiceId = $config['elevenlabs']['voice_id'] ?? '';
        
        if (empty($this->apiKey)) {
            throw new Exception('Elevenlabs API anahtarı bulunamadı. Lütfen ELEVENLABS_API_KEY değişkenini ayarlayın.');
        }
        
        if (empty($this->agentId)) {
            throw new Exception('Elevenlabs Agent ID bulunamadı. Lütfen ELEVENLABS_AGENT_ID değişkenini ayarlayın.');
        }
    }
    
    /**
     * Create or get conversational agent
     */
    public function createAgent($knowledgeBase = [])
    {
        $assistantConfig = $this->config['assistant'];
        
        $data = [
            'name' => $assistantConfig['name'],
            'voice_id' => $this->voiceId,
            'prompt' => [
                'prompt' => $this->generateSystemPrompt()
            ],
            'first_message' => $this->generateGreeting('professional'),
            'language' => 'tr'  // Turkish
        ];
        
        return $this->makeRequest('POST', '/convai/agents', $data);
    }
    
    /**
     * Update agent configuration
     */
    public function updateAgent($agentId, $updates)
    {
        return $this->makeRequest('PATCH', "/convai/agents/{$agentId}", $updates);
    }
    
    /**
     * Start a conversation with the agent (WebSocket simulation)
     * Bu fonksiyon WebSocket bağlantısını simüle eder
     */
    public function startConversation($customerName = null, $customerPhone = null, $context = [])
    {
        if (empty($this->agentId)) {
            return [
                'success' => false,
                'error' => 'Agent ID yapılandırılmamış'
            ];
        }
        
        if (empty($customerPhone)) {
            return [
                'success' => false,
                'error' => 'Müşteri telefon numarası gerekli'
            ];
        }
        
        try {
            // Telefon numarasını formatla
            $formattedPhone = $this->formatTurkishPhoneNumber($customerPhone);
            
            // Gerçek telefon araması başlat
            $result = $this->initiateOutboundCall($customerName, $formattedPhone, $context);
            
            return [
                'success' => true,
                'conversation_id' => $result['conversation_id'],
                'call_id' => $result['call_id'] ?? null,
                'greeting' => $this->generateGreeting('professional', $customerName),
                'customer_info' => [
                    'name' => $customerName,
                    'phone' => $formattedPhone,
                    'context' => $context
                ],
                'message' => 'Telefon araması başlatıldı'
            ];
            
        } catch (Exception $e) {
            error_log('Phone call start error: ' . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Telefon araması başlatılamadı: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Initiate outbound phone call using Elevenlabs Batch Calling API
     */
    public function initiateOutboundCall($customerName = null, $customerPhone = null, $context = [])
    {
        // Phone number ID'sini konfigürasyondan al
        $phoneNumberId = $this->config['elevenlabs']['phone_number_id'] ?? null;
        
        if (empty($phoneNumberId)) {
            throw new Exception('Telefon numarası ID\'si yapılandırılmamış. Lütfen ELEVENLABS_PHONE_NUMBER_ID ayarını kontrol edin.');
        }
        
        // Telefon numarasını Türk formatına çevir
        $customerPhone = $this->formatTurkishPhoneNumber($customerPhone);
        
        // Prepare dynamic variables with customer context
        $dynamicVariables = [
            'musteri_adi' => $customerName ?: 'Müşteri'
        ];
        
        // Add context information as dynamic variables
        if (!empty($context)) {
            if (!empty($context['interest'])) {
                $dynamicVariables['ilgi_alani'] = $context['interest'];
            }
            if (!empty($context['budget'])) {
                $dynamicVariables['butce'] = $context['budget'];
            }
            if (!empty($context['location'])) {
                $dynamicVariables['bolge'] = $context['location'];
            }
            if (!empty($context['urgency'])) {
                $dynamicVariables['aciliyet'] = $context['urgency'];
            }
            if (!empty($context['notes'])) {
                $dynamicVariables['notlar'] = $context['notes'];
            }
        }
        
        // Batch call için veri hazırla (hemen başlatmak için mevcut timestamp)
        $currentTime = time();
        
        $batchData = [
            'call_name' => 'Gayrimenkul AI - ' . ($customerName ?: 'Müşteri') . ' - ' . date('Y-m-d H:i:s'),
            'agent_id' => $this->agentId,
            'agent_phone_number_id' => $phoneNumberId,
            'recipients' => [
                [
                    'phone_number' => $customerPhone,
                    'dynamic_variables' => $dynamicVariables
                ]
            ],
            'scheduled_time_unix' => $currentTime // Hemen başlatmak için mevcut timestamp
        ];
        
        // Debug: Log dynamic variables
        error_log('Dynamic Variables being sent: ' . json_encode($dynamicVariables, JSON_UNESCAPED_UNICODE));
        error_log('Batch Data: ' . json_encode($batchData, JSON_UNESCAPED_UNICODE));
        
        // API isteği gönder
        $response = $this->makeRequest('POST', '/convai/batch-calling/submit', $batchData);
        
        if (!$response || !isset($response['id'])) {
            throw new Exception('Batch call oluşturulamadı: ' . ($response['detail']['message'] ?? 'Bilinmeyen hata'));
        }
        
        return [
            'conversation_id' => 'batch_' . $response['id'],
            'batch_id' => $response['id'],
            'call_id' => $response['id'],
            'status' => $response['status'] ?? 'initiated',
            'total_calls_scheduled' => $response['total_calls_scheduled'] ?? 1,
            'dynamic_variables' => $dynamicVariables
        ];
    }
    
    /**
     * Send message to conversation (simulated for WebSocket)
     */
    public function sendMessage($conversationId, $message, $customerInfo = [])
    {
        // WebSocket tabanlı sistem için simüle edilmiş yanıt
        $responses = [
            'Merhaba' => 'Merhaba! Ben Ayşe Kaya, Elite Gayrimenkul\'den. Size nasıl yardımcı olabilirim?',
            'daire arıyorum' => 'Tabii, size uygun daire seçeneklerini bulabilirim. Kaç odalı bir daire arıyorsunuz ve hangi bölgeyi tercih edersiniz?',
            '2+1' => 'Harika! 2+1 daireler için çok güzel seçeneklerimiz var. Bütçeniz nedir ve hangi semtleri düşünüyorsunuz?',
            'default' => 'Anlıyorum. Bu konuda size yardımcı olmaya çalışayım. Daha detaylı bilgi verebilir misiniz?'
        ];
        
        $response = $responses['default'];
        foreach ($responses as $key => $value) {
            if ($key !== 'default' && stripos($message, $key) !== false) {
                $response = $value;
                break;
            }
        }
        
        return [
            'success' => true,
            'response' => $response,
            'conversation_id' => $conversationId
        ];
    }
    
    /**
     * Get conversation history
     */
    public function getConversationHistory($conversationId)
    {
        return [
            'success' => true,
            'messages' => [],
            'note' => 'WebSocket tabanlı sistem - geçmiş mesajlar WebSocket üzerinden yönetilir'
        ];
    }
    
    /**
     * End conversation
     */
    public function endConversation($conversationId)
    {
        return [
            'success' => true,
            'message' => 'Konuşma sonlandırıldı'
        ];
    }
    
    /**
     * Add knowledge base content
     */
    public function addKnowledgeBase($content, $type = 'text', $category = 'general')
    {
        $data = [
            'name' => $category,
            'content' => $content,
            'type' => $type
        ];
        
        return $this->makeRequest('POST', '/convai/agents/' . $this->agentId . '/knowledge-base', $data);
    }
    
    /**
     * Update knowledge base content
     */
    public function updateKnowledgeBase($knowledgeId, $content)
    {
        return $this->makeRequest('PATCH', "/convai/agents/{$this->agentId}/knowledge-base/{$knowledgeId}", [
            'content' => $content
        ]);
    }
    
    /**
     * Delete knowledge base content
     */
    public function deleteKnowledgeBase($knowledgeId)
    {
        return $this->makeRequest('DELETE', "/convai/agents/{$this->agentId}/knowledge-base/{$knowledgeId}");
    }
    
    /**
     * Get available voices
     */
    public function getVoices()
    {
        return $this->makeRequest('GET', '/voices');
    }
    
    /**
     * Generate system prompt based on configuration
     */
    private function generateSystemPrompt()
    {
        $assistant = $this->config['assistant'];
        $company = $assistant['company'];
        $name = $assistant['name'];
        $role = $assistant['role'];
        
        $prompt = "Sen {$name} adında bir {$role}sın ve {$company} şirketinde çalışıyorsun. ";
        $prompt .= "Gayrimenkul konularında uzman bir danışman olarak müşterilere yardımcı oluyorsun.\n\n";
        
        $prompt .= "GÖREVLERİN:\n";
        $prompt .= "- Müşterilerin gayrimenkul ihtiyaçlarını anlayıp en uygun çözümleri sunmak\n";
        $prompt .= "- Satış ve kiralama süreçleri hakkında detaylı bilgi vermek\n";
        $prompt .= "- Mortgage ve kredi konularında rehberlik etmek\n";
        $prompt .= "- Yatırım fırsatları hakkında danışmanlık yapmak\n";
        $prompt .= "- Bölge analizleri ve piyasa durumu hakkında bilgilendirmek\n\n";
        
        $prompt .= "KONUŞMA TARZI:\n";
        $prompt .= "- Profesyonel ama samimi bir dil kullan\n";
        $prompt .= "- Kısa ve anlaşılır cümleler kur\n";
        $prompt .= "- Müşterinin adını öğrenmeye çalış ve konuşmada kullan\n";
        $prompt .= "- Soru sormaktan çekinme, detayları öğren\n";
        $prompt .= "- Her zaman yardımcı olmaya odaklan\n\n";
        
        $prompt .= "ÖNEMLİ KURALLAR:\n";
        $prompt .= "- İlk konuşmada kendini tanıt\n";
        $prompt .= "- Bilmediğin konularda spekülasyon yapma\n";
        $prompt .= "- Müşteri bilgilerini güvenli tut\n";
        $prompt .= "- Randevu talepleri için uygun zamanları sor\n";
        $prompt .= "- Fiyat verirken güncel piyasa verilerini kullan\n\n";
        
        $prompt .= "Şimdi müşterilerle profesyonel ve yardımsever bir şekilde konuş. Türkçe yanıt ver.";
        
        return $prompt;
    }
    
    /**
     * Generate greeting message
     */
    public function generateGreeting($style = 'professional', $customerName = null)
    {
        $assistant = $this->config['assistant'];
        $greetings = isset($assistant['greeting']) ? $assistant['greeting'] : [
            'professional' => 'Merhaba, ben {name}, {company} şirketinden. Size nasıl yardımcı olabilirim?',
            'friendly' => 'Merhabalar! Ben {name}. {company}\'dan arıyorum. Nasılsınız?',
            'formal' => 'İyi günler. Ben {name}, {company} uzmanıyım. Size nasıl yardımcı olabilirim?'
        ];
        
        $template = isset($greetings[$style]) ? $greetings[$style] : $greetings['professional'];
        
        $greeting = str_replace(
            ['{name}', '{company}'],
            [$assistant['name'], $assistant['company']],
            $template
        );
        
        if ($customerName) {
            $greeting = str_replace(
                ['Size nasıl', 'size nasıl'],
                ["{$customerName} bey/hanım, size nasıl", "{$customerName} bey/hanım, size nasıl"],
                $greeting
            );
        }
        
        return $greeting;
    }
    
    /**
     * Make HTTP request to Elevenlabs API
     */
    private function makeRequest($method, $endpoint, $data = null)
    {
        $url = $this->baseUrl . $endpoint;
        
        $headers = [
            'Content-Type: application/json',
            'xi-api-key: ' . $this->apiKey,
            'User-Agent: GayrimenkulAI/1.0'
        ];
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_TIMEOUT, $this->config['elevenlabs']['timeout'] ?? 30);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        
        if ($method === 'POST') {
            curl_setopt($ch, CURLOPT_POST, true);
            if ($data) {
                curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
            }
        } elseif ($method === 'PATCH') {
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'PATCH');
            if ($data) {
                curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
            }
        } elseif ($method === 'DELETE') {
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'DELETE');
        }
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);
        
        if ($error) {
            throw new Exception('cURL Error: ' . $error);
        }
        
        $decodedResponse = json_decode($response, true);
        
        if ($httpCode >= 400) {
            $errorMessage = isset($decodedResponse['detail']['message']) ? 
                           $decodedResponse['detail']['message'] : 
                           (isset($decodedResponse['message']) ? $decodedResponse['message'] : 'Bilinmeyen API hatası');
            throw new Exception("Elevenlabs API Hatası ({$httpCode}): " . $errorMessage);
        }
        
        return $decodedResponse;
    }
    
    /**
     * Validate configuration
     */
    public function validateConfiguration()
    {
        $errors = [];
        
        if (empty($this->apiKey)) {
            $errors[] = 'API anahtarı eksik';
        }
        
        if (empty($this->baseUrl)) {
            $errors[] = 'Base URL eksik';
        }
        
        if (empty($this->agentId)) {
            $errors[] = 'Agent ID eksik';
        }
        
        return $errors;
    }
    
    /**
     * Get WebSocket connection URL for conversational AI
     */
    public function getWebSocketUrl()
    {
        return $this->wsUrl . '?agent_id=' . $this->agentId;
    }
    
    /**
     * Generate WebSocket initialization data
     */
    public function getWebSocketInitData($customerName = null, $customerPhone = null)
    {
        $assistantConfig = $this->config['assistant'];
        
        return [
            'type' => 'conversation_initiation_client_data',
            'conversation_config_override' => [
                'agent' => [
                    'prompt' => [
                        'prompt' => $this->generateSystemPrompt()
                    ],
                    'first_message' => $this->generateGreeting('professional', $customerName),
                    'language' => 'tr'
                ],
                'tts' => [
                    'voice_id' => $this->voiceId
                ]
            ],
            'dynamic_variables' => [
                'user_name' => $customerName ?: 'Müşteri',
                'user_phone' => $customerPhone ?: '',
                'company' => $assistantConfig['company'],
                'agent_name' => $assistantConfig['name']
            ]
        ];
    }
    
    /**
     * Format Turkish phone number to international format
     */
    public function formatTurkishPhoneNumber($phone)
    {
        if (empty($phone)) {
            return $phone;
        }
        
        // Sadece rakamları ve + işaretini bırak
        $cleaned = preg_replace('/[^\d+]/', '', $phone);
        
        // Türk telefon numarası formatlarını işle
        if (preg_match('/^0(\d{10})$/', $cleaned, $matches)) {
            // 0 ile başlıyorsa +90 ile değiştir
            return '+90' . $matches[1];
        } elseif (preg_match('/^90(\d{10})$/', $cleaned, $matches)) {
            // 90 ile başlıyorsa + ekle
            return '+' . $cleaned;
        } elseif (preg_match('/^\d{10}$/', $cleaned)) {
            // 10 haneli numara ise +90 ekle
            return '+90' . $cleaned;
        } elseif (preg_match('/^\+90\d{10}$/', $cleaned)) {
            // Zaten doğru formatta
            return $cleaned;
        }
        
        // Diğer durumlarda orijinal numarayı döndür
        return $phone;
    }

    /**
     * Batch call durumunu kontrol et
     */
    public function getBatchCallStatus($batchCallId) {
        $url = $this->baseUrl . '/convai/batch-calling/' . $batchCallId;
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'xi-api-key: ' . $this->apiKey,
            'Content-Type: application/json'
        ]);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode !== 200) {
            throw new Exception("Batch call status check failed: HTTP $httpCode");
        }
        
        $data = json_decode($response, true);
        return $data;
    }

    /**
     * Conversation transcript'ini al
     */
    public function getConversationTranscript($conversationId) {
        $url = $this->baseUrl . '/convai/conversations/' . $conversationId;
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'xi-api-key: ' . $this->apiKey,
            'Content-Type: application/json'
        ]);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode !== 200) {
            throw new Exception("Conversation transcript fetch failed: HTTP $httpCode");
        }
        
        $data = json_decode($response, true);
        return $data;
    }
} 