<?php

try {
    $input = json_decode(file_get_contents('php://input'), true);
    
    // Initialize services
    $elevenlabsService = new ElevenlabsService($config);
    $knowledgeManager = new KnowledgeManager($config, $elevenlabsService);
    
    switch ($_SERVER['REQUEST_METHOD']) {
        case 'POST':
            $action = $input['action'] ?? ($_SERVER['REQUEST_URI'] === '/api/knowledge/initialize' ? 'initialize' : 'add');
            
            switch ($action) {
                case 'add':
                    // Add knowledge content
                    $content = $input['content'] ?? '';
                    $category = $input['category'] ?? 'genel_bilgiler';
                    $title = $input['title'] ?? '';
                    $type = $input['type'] ?? 'text';
                    $source = $input['source'] ?? null;
                    
                    if (empty($content) || empty($title)) {
                        throw new Exception('İçerik ve başlık gerekli');
                    }
                    
                    $result = $knowledgeManager->addContent($content, $category, $title, $type, $source);
                    echo json_encode($result);
                    break;
                    
                case 'upload':
                    // Upload file
                    if (!isset($_FILES['file'])) {
                        throw new Exception('Dosya gerekli');
                    }
                    
                    $file = $_FILES['file'];
                    $category = $_POST['category'] ?? 'genel_bilgiler';
                    $title = $_POST['title'] ?? null;
                    
                    $result = $knowledgeManager->uploadFile($file, $category, $title);
                    echo json_encode($result);
                    break;
                    
                case 'url':
                    // Add URL content
                    $url = $input['url'] ?? '';
                    $category = $input['category'] ?? 'genel_bilgiler';
                    $title = $input['title'] ?? null;
                    
                    if (empty($url)) {
                        throw new Exception('URL gerekli');
                    }
                    
                    $result = $knowledgeManager->addUrlContent($url, $category, $title);
                    echo json_encode($result);
                    break;
                    
                case 'initialize':
                    // Initialize default knowledge base
                    $result = $knowledgeManager->initializeDefaults();
                    
                    $successCount = 0;
                    foreach ($result as $res) {
                        if ($res['success']) $successCount++;
                    }
                    
                    echo json_encode([
                        'success' => $successCount > 0,
                        'results' => $result,
                        'message' => "{$successCount} kategori başarıyla yüklendi"
                    ]);
                    break;
                    
                default:
                    throw new Exception('Geçersiz action');
            }
            break;
            
        case 'GET':
            $action = $_GET['action'] ?? 'list';
            
            switch ($action) {
                case 'list':
                    // List all knowledge base content
                    $category = $_GET['category'] ?? null;
                    $content = $knowledgeManager->getAllContent($category);
                    
                    echo json_encode([
                        'success' => true,
                        'knowledge_base' => $content,
                        'categories' => $config['knowledge_base']['categories']
                    ]);
                    break;
                    
                case 'search':
                    // Search in knowledge base
                    $query = $_GET['query'] ?? '';
                    $category = $_GET['category'] ?? null;
                    
                    if (empty($query)) {
                        throw new Exception('Arama sorgusu gerekli');
                    }
                    
                    $results = $knowledgeManager->searchContent($query, $category);
                    
                    echo json_encode([
                        'success' => true,
                        'results' => $results,
                        'query' => $query
                    ]);
                    break;
                    
                case 'templates':
                    // Get knowledge base templates
                    $templates = $knowledgeManager->getTemplates();
                    
                    echo json_encode([
                        'success' => true,
                        'templates' => $templates
                    ]);
                    break;
                    
                default:
                    throw new Exception('Geçersiz action');
            }
            break;
            
        case 'PUT':
            // Update knowledge content
            $contentId = $input['content_id'] ?? '';
            $newContent = $input['content'] ?? '';
            
            if (empty($contentId) || empty($newContent)) {
                throw new Exception('Content ID ve yeni içerik gerekli');
            }
            
            $result = $knowledgeManager->updateContent($contentId, $newContent);
            echo json_encode($result);
            break;
            
        case 'DELETE':
            // Delete knowledge content
            $contentId = $input['content_id'] ?? $_GET['content_id'] ?? '';
            
            if (empty($contentId)) {
                throw new Exception('Content ID gerekli');
            }
            
            $result = $knowledgeManager->deleteContent($contentId);
            echo json_encode($result);
            break;
            
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
    
} catch (Exception $e) {
    error_log('Knowledge API Error: ' . $e->getMessage());
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}
?> 