<?php
require_once __DIR__ . '/../backend/ElevenlabsService.php';
require_once __DIR__ . '/../config/config.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

$elevenlabsService = new ElevenlabsService($config);

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $input = json_decode(file_get_contents('php://input'), true);
    
    $customerName = $input['customer_name'] ?? '';
    $customerPhone = $input['customer_phone'] ?? '';
    $customerContext = $input['customer_context'] ?? [];
    
    try {
        $result = $elevenlabsService->startConversation($customerName, $customerPhone, $customerContext);
        
        $response = [
            'success' => true,
            'conversation_id' => $result['conversation_id'],
            'greeting' => $elevenlabsService->generateGreeting('professional', $customerName),
            'customer_info' => [
                'name' => $customerName,
                'phone' => $customerPhone,
                'context' => $customerContext
            ],
            'message' => 'Telefon araması başlatıldı',
            'call_id' => $result['call_id'],
            'status' => $result['status'] ?? 'initiated',
            'monitoring' => [
                'status_check_url' => '/api/conversation?action=status&call_id=' . $result['call_id'],
                'transcript_url' => '/api/conversation?action=transcript&conversation_id=' . $result['conversation_id'],
                'realtime_url' => '/api/conversation?action=monitor&call_id=' . $result['call_id']
            ]
        ];
        
        echo json_encode($response);
        
    } catch (Exception $e) {
        error_log("Phone call start error: " . $e->getMessage());
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'error' => $e->getMessage()
        ]);
    }
} 

// Real-time conversation monitoring
elseif ($_SERVER['REQUEST_METHOD'] === 'GET' && isset($_GET['action']) && $_GET['action'] === 'monitor') {
    try {
        $callId = $_GET['call_id'] ?? '';
        if (empty($callId)) {
            throw new Exception('Call ID gerekli');
        }
        
        // Real-time monitoring data
        $batchCallId = str_replace('batch_', '', $callId);
        $status = $elevenlabsService->getBatchCallStatus($batchCallId);
        
        // Get detailed monitoring info
        $monitoringData = [
            'call_id' => $callId,
            'status' => $status['status'] ?? 'unknown',
            'timestamp' => time(),
            'recipients' => $status['recipients'] ?? [],
            'conversation_ids' => [],
            'live_transcript' => null
        ];
        
        // Extract conversation IDs from recipients
        if (isset($status['recipients']) && is_array($status['recipients'])) {
            foreach ($status['recipients'] as $recipient) {
                if (isset($recipient['conversation_id'])) {
                    $monitoringData['conversation_ids'][] = $recipient['conversation_id'];
                }
            }
        }
        
        // If call is completed, get transcript
        if ($status['status'] === 'completed' && !empty($monitoringData['conversation_ids'])) {
            foreach ($monitoringData['conversation_ids'] as $convId) {
                try {
                    $transcript = $elevenlabsService->getConversationTranscript($convId);
                    $monitoringData['live_transcript'] = $transcript;
                    break; // Get first available transcript
                } catch (Exception $e) {
                    // Continue to next conversation
                }
            }
        }
        
        echo json_encode([
            'success' => true,
            'monitoring' => $monitoringData
        ]);
        
    } catch (Exception $e) {
        error_log("Real-time monitoring error: " . $e->getMessage());
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'error' => $e->getMessage()
        ]);
    }
}

// Batch call durumunu kontrol et  
elseif ($_SERVER['REQUEST_METHOD'] === 'GET' && isset($_GET['action']) && $_GET['action'] === 'status') {
    try {
        $callId = $_GET['call_id'] ?? '';
        if (empty($callId)) {
            throw new Exception('Call ID gerekli');
        }
        
        // Batch call ID'sinden "batch_" prefix'ini temizle
        $batchCallId = str_replace('batch_', '', $callId);
        
        $status = $elevenlabsService->getBatchCallStatus($batchCallId);
        
        echo json_encode([
            'success' => true,
            'status' => $status['status'] ?? null,
            'details' => $status
        ]);
        
    } catch (Exception $e) {
        error_log("Status check error: " . $e->getMessage());
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'error' => $e->getMessage()
        ]);
    }
}

// Conversation transcript'lerini al
elseif ($_SERVER['REQUEST_METHOD'] === 'GET' && isset($_GET['action']) && $_GET['action'] === 'transcript') {
    try {
        $conversationId = $_GET['conversation_id'] ?? '';
        if (empty($conversationId)) {
            throw new Exception('Conversation ID gerekli');
        }
        
        $transcript = $elevenlabsService->getConversationTranscript($conversationId);
        
        echo json_encode([
            'success' => true,
            'transcript' => $transcript
        ]);
        
    } catch (Exception $e) {
        error_log("Transcript fetch error: " . $e->getMessage());
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'error' => $e->getMessage()
        ]);
    }
}

// NEW: Webhook endpoint for Elevenlabs conversation initiation
elseif ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_GET['webhook'])) {
    try {
        $webhookData = json_decode(file_get_contents('php://input'), true);
        
        // Log webhook data for debugging
        error_log("Elevenlabs Webhook received: " . json_encode($webhookData, JSON_UNESCAPED_UNICODE));
        
        // Store webhook data
        $webhookFile = __DIR__ . '/../data/webhooks.log';
        if (!file_exists(dirname($webhookFile))) {
            mkdir(dirname($webhookFile), 0755, true);
        }
        file_put_contents($webhookFile, 
            date('Y-m-d H:i:s') . " - " . json_encode($webhookData, JSON_UNESCAPED_UNICODE) . "\n", 
            FILE_APPEND | LOCK_EX
        );
        
        // Check if this is a conversation initiation request
        if (isset($webhookData['type']) || 
            (isset($webhookData['event_type']) && $webhookData['event_type'] === 'conversation_initiation') ||
            (isset($webhookData['data']) && isset($webhookData['data']['dynamic_variables']))) {
            
            // Return conversation_initiation_client_data with dynamic variables
            $clientData = [
                'type' => 'conversation_initiation_client_data',
                'conversation_config_override' => [
                    'agent' => [
                        'prompt' => [
                            'prompt' => file_get_contents(__DIR__ . '/../../system_prompt_template.txt')
                        ],
                        'first_message' => file_get_contents(__DIR__ . '/../../first_message_template.txt'),
                        'language' => 'tr'
                    ]
                ],
                'dynamic_variables' => [
                    'musteri_adi' => 'Ahmet Özkan',
                    'ilgi_alani' => '3+1 daire', 
                    'butce' => '1M-2M TL',
                    'bolge' => 'Kadıköy, Beşiktaş',
                    'aciliyet' => 'normal',
                    'notlar' => 'Deniz manzaralı tercih ediyor, yatırım amaçlı'
                ]
            ];
            
            error_log("Sending conversation_initiation_client_data: " . json_encode($clientData, JSON_UNESCAPED_UNICODE));
            echo json_encode($clientData);
        } else {
            // Regular webhook response
            echo json_encode([
                'success' => true,
                'message' => 'Webhook received'
            ]);
        }
        
    } catch (Exception $e) {
        error_log("Webhook processing error: " . $e->getMessage());
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'error' => $e->getMessage()
        ]);
    }
}

else {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
}
?> 